<?php
namespace Services;

use Utils\Database;

class StorageService {
    public static function createRoom($name, $facilityAddress, $managerId = null) {
        $db = Database::getInstance();

        return $db->execute(
            "INSERT INTO rooms (name, facility_address, manager_id) VALUES (?, ?, ?)",
            [$name, $facilityAddress, $managerId]
        );
    }

    public static function createStorageUnit($roomId, $name, $type, $tempRange, $dimensions, $capacity) {
        $db = Database::getInstance();

        return $db->execute(
            "INSERT INTO storage_units (room_id, name, type, temp_range, dimensions_rows, dimensions_cols, dimensions_levels, capacity)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [$roomId, $name, $type, $tempRange, $dimensions['rows'], $dimensions['cols'], $dimensions['levels'], $capacity]
        );
    }

    public static function createShelf($storageUnitId, $name, $identifier) {
        $db = Database::getInstance();

        return $db->execute(
            "INSERT INTO shelves (storage_unit_id, name, identifier) VALUES (?, ?, ?)",
            [$storageUnitId, $name, $identifier]
        );
    }

    public static function createBox($shelfId, $name, $rows, $columns) {
        $db = Database::getInstance();

        $capacity = $rows * $columns;

        $boxId = $db->execute(
            "INSERT INTO boxes (shelf_id, name, box_rows, box_columns, capacity) VALUES (?, ?, ?, ?, ?)",
            [$shelfId, $name, $rows, $columns, $capacity]
        );

        // Create positions for this box
        for ($row = 1; $row <= $rows; $row++) {
            for ($col = 1; $col <= $columns; $col++) {
                $db->execute(
                    "INSERT INTO positions (box_id, row_index, col_index, status) VALUES (?, ?, ?, 'empty')",
                    [$boxId, $row, $col]
                );
            }
        }

        return $boxId;
    }

    public static function getStorageHierarchy() {
        $db = Database::getInstance();

        return $db->fetchAll("
            SELECT
                r.id as room_id, r.name as room_name,
                su.id as storage_unit_id, su.name as storage_unit_name, su.type as storage_unit_type,
                s.id as shelf_id, s.name as shelf_name, s.identifier as shelf_identifier,
                b.id as box_id, b.name as box_name, b.box_rows, b.box_columns,
                p.id as position_id, p.row_index, p.col_index, p.status
            FROM rooms r
            LEFT JOIN storage_units su ON r.id = su.room_id
            LEFT JOIN shelves s ON su.id = s.storage_unit_id
            LEFT JOIN boxes b ON s.id = b.shelf_id
            LEFT JOIN positions p ON b.id = p.box_id
            ORDER BY r.name, su.name, s.name, b.name, p.row_index, p.col_index
        ");
    }

    public static function getPositionDetails($positionId) {
        $db = Database::getInstance();

        return $db->fetch("
            SELECT p.*, b.name as box_name, s.name as shelf_name, su.name as storage_unit_name, r.name as room_name
            FROM positions p
            JOIN boxes b ON p.box_id = b.id
            JOIN shelves s ON b.shelf_id = s.id
            JOIN storage_units su ON s.storage_unit_id = su.id
            JOIN rooms r ON su.room_id = r.id
            WHERE p.id = ?
        ", [$positionId]);
    }

    public static function storeSpecimen($barcode, $clientId, $positionId, $sampleType, $volume, $diseaseOntology = null) {
        $db = Database::getInstance();

        // Check if position is available
        $position = $db->fetch("SELECT status FROM positions WHERE id = ?", [$positionId]);
        if ($position['status'] !== 'empty') {
            throw new \Exception("Position is not available");
        }

        // Mark position as occupied
        $db->execute("UPDATE positions SET status = 'occupied' WHERE id = ?", [$positionId]);

        // Create specimen
        return $db->execute(
            "INSERT INTO specimens (barcode, client_id, box_position_id, sample_type, volume, disease_ontology)
             VALUES (?, ?, ?, ?, ?, ?)",
            [$barcode, $clientId, $positionId, $sampleType, $volume, $diseaseOntology]
        );
    }

    public static function searchSpecimen($barcode) {
        $db = Database::getInstance();

        $specimen = $db->fetch("
            SELECT s.*, p.row_index, p.col_index, b.name as box_name, s2.name as shelf_name,
                   su.name as storage_unit_name, r.name as room_name
            FROM specimens s
            JOIN positions p ON s.box_position_id = p.id
            JOIN boxes b ON p.box_id = b.id
            JOIN shelves s2 ON b.shelf_id = s2.id
            JOIN storage_units su ON s2.storage_unit_id = su.id
            JOIN rooms r ON su.room_id = r.id
            WHERE s.barcode = ?
        ", [$barcode]);

        if (!$specimen) {
            return null;
        }

        return [
            'specimen' => $specimen,
            'location' => "{$specimen['room_name']} > {$specimen['storage_unit_name']} > {$specimen['shelf_name']} > {$specimen['box_name']} > Position {$specimen['row_index']}-{$specimen['col_index']}"
        ];
    }

    public static function getClientSpecimens($clientId) {
        $db = Database::getInstance();

        return $db->fetchAll("
            SELECT s.*, p.row_index, p.col_index, b.name as box_name
            FROM specimens s
            JOIN positions p ON s.box_position_id = p.id
            JOIN boxes b ON p.box_id = b.id
            WHERE s.client_id = ?
            ORDER BY s.received_at DESC
        ", [$clientId]);
    }
}
