<?php
namespace Services;

use Utils\Database;

class DatabaseSetup {
    public static function createTables() {
        $db = Database::getInstance();

        // Create storage hierarchy tables
        $tables = [
            "CREATE TABLE IF NOT EXISTS rooms (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                facility_address TEXT NOT NULL,
                manager_id INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )",

            "CREATE TABLE IF NOT EXISTS storage_units (
                id INT AUTO_INCREMENT PRIMARY KEY,
                room_id INT NOT NULL,
                name VARCHAR(100) NOT NULL,
                type ENUM('-80', '-20', 'LN2') NOT NULL,
                temp_range VARCHAR(50),
                dimensions_rows INT,
                dimensions_cols INT,
                dimensions_levels INT,
                capacity INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (room_id) REFERENCES rooms(id) ON DELETE CASCADE
            )",

            "CREATE TABLE IF NOT EXISTS shelves (
                id INT AUTO_INCREMENT PRIMARY KEY,
                storage_unit_id INT NOT NULL,
                name VARCHAR(100) NOT NULL,
                identifier VARCHAR(50),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (storage_unit_id) REFERENCES storage_units(id) ON DELETE CASCADE
            )",

            "CREATE TABLE IF NOT EXISTS boxes (
                id INT AUTO_INCREMENT PRIMARY KEY,
                shelf_id INT NOT NULL,
                name VARCHAR(100) NOT NULL,
                box_rows INT NOT NULL,
                box_columns INT NOT NULL,
                capacity INT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (shelf_id) REFERENCES shelves(id) ON DELETE CASCADE
            )",

            "CREATE TABLE IF NOT EXISTS positions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                box_id INT NOT NULL,
                row_index INT NOT NULL,
                col_index INT NOT NULL,
                status ENUM('empty', 'occupied', 'reserved') DEFAULT 'empty',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (box_id) REFERENCES boxes(id) ON DELETE CASCADE,
                UNIQUE KEY (box_id, row_index, col_index)
            )",

            // Specimen table
            "CREATE TABLE IF NOT EXISTS specimens (
                id INT AUTO_INCREMENT PRIMARY KEY,
                barcode VARCHAR(100) UNIQUE NOT NULL,
                client_id INT NOT NULL,
                box_position_id INT,
                sample_type VARCHAR(100) NOT NULL,
                volume DECIMAL(10,2),
                disease_ontology TEXT,
                received_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                shipped_at TIMESTAMP NULL,
                status ENUM('Stored', 'Shipped', 'Destroyed') DEFAULT 'Stored',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (box_position_id) REFERENCES positions(id) ON DELETE SET NULL,
                FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE CASCADE
            )",

            // Billing tables
            "CREATE TABLE IF NOT EXISTS billing_rates (
                id INT AUTO_INCREMENT PRIMARY KEY,
                sample_type VARCHAR(100) NOT NULL,
                daily_rate DECIMAL(10,2) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY (sample_type)
            )",

            "CREATE TABLE IF NOT EXISTS additional_services (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                base_cost DECIMAL(10,2) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )",

            "CREATE TABLE IF NOT EXISTS invoices (
                id INT AUTO_INCREMENT PRIMARY KEY,
                client_id INT NOT NULL,
                total_amount DECIMAL(10,2) NOT NULL,
                status ENUM('Pending', 'Paid') DEFAULT 'Pending',
                period_start DATE NOT NULL,
                period_end DATE NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE CASCADE
            )",

            // User and authentication tables
            "CREATE TABLE IF NOT EXISTS roles (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(50) NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY (name)
            )",

            "CREATE TABLE IF NOT EXISTS clients (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                email VARCHAR(100) NOT NULL,
                phone VARCHAR(20),
                address TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY (email)
            )",

            "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                email VARCHAR(100) NOT NULL,
                password VARCHAR(255) NOT NULL,
                role_id INT NOT NULL,
                client_id INT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY (email),
                FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
                FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE SET NULL
            )"
        ];

        foreach ($tables as $table) {
            $db->execute($table);
        }

        // Insert default roles
        $roles = ['Admin', 'Staff', 'Client'];
        foreach ($roles as $role) {
            $db->execute("INSERT IGNORE INTO roles (name) VALUES (?)", [$role]);
        }

        // Insert some default billing rates
        $defaultRates = [
            ['Plasma', 1.00],
            ['Tissue', 2.00],
            ['Blood', 1.50],
            ['DNA', 2.50],
            ['RNA', 2.50]
        ];

        foreach ($defaultRates as $rate) {
            $db->execute("INSERT IGNORE INTO billing_rates (sample_type, daily_rate) VALUES (?, ?)", $rate);
        }

        return true;
    }
}
