<?php
namespace Services;

use Utils\Database;

class BillingService {
    public static function calculateSpecimenCost($specimenId) {
        $db = Database::getInstance();

        // Get specimen details
        $specimen = $db->fetch("
            SELECT s.*, b.daily_rate
            FROM specimens s
            JOIN billing_rates b ON s.sample_type = b.sample_type
            WHERE s.id = ?
        ", [$specimenId]);

        if (!$specimen) {
            throw new \Exception("Specimen not found or no billing rate defined for this sample type");
        }

        // Calculate duration in days
        $receivedAt = new \DateTime($specimen['received_at']);
        $now = new \DateTime();
        $days = $receivedAt->diff($now)->days;

        // Calculate cost
        $cost = $days * $specimen['daily_rate'];

        return [
            'specimen_id' => $specimen['id'],
            'barcode' => $specimen['barcode'],
            'sample_type' => $specimen['sample_type'],
            'days_stored' => $days,
            'daily_rate' => $specimen['daily_rate'],
            'total_cost' => $cost
        ];
    }

    public static function calculateClientBalance($clientId) {
        $db = Database::getInstance();

        // Get all active specimens for this client
        $specimens = $db->fetchAll("
            SELECT s.*, b.daily_rate
            FROM specimens s
            JOIN billing_rates b ON s.sample_type = b.sample_type
            WHERE s.client_id = ? AND s.status = 'Stored'
        ", [$clientId]);

        $totalBalance = 0;
        $specimenDetails = [];

        foreach ($specimens as $specimen) {
            $receivedAt = new \DateTime($specimen['received_at']);
            $now = new \DateTime();
            $days = $receivedAt->diff($now)->days;
            $cost = $days * $specimen['daily_rate'];

            $totalBalance += $cost;

            $specimenDetails[] = [
                'specimen_id' => $specimen['id'],
                'barcode' => $specimen['barcode'],
                'sample_type' => $specimen['sample_type'],
                'days_stored' => $days,
                'daily_rate' => $specimen['daily_rate'],
                'cost' => $cost
            ];
        }

        return [
            'client_id' => $clientId,
            'total_balance' => $totalBalance,
            'specimens' => $specimenDetails
        ];
    }

    public static function generateInvoice($clientId, $periodStart, $periodEnd) {
        $db = Database::getInstance();

        // Calculate balance for the period
        $balance = self::calculateClientBalance($clientId);

        // Create invoice
        $invoiceId = $db->execute(
            "INSERT INTO invoices (client_id, total_amount, period_start, period_end) VALUES (?, ?, ?, ?)",
            [$clientId, $balance['total_balance'], $periodStart, $periodEnd]
        );

        return $invoiceId;
    }

    public static function getClientInvoices($clientId) {
        $db = Database::getInstance();

        return $db->fetchAll("
            SELECT * FROM invoices
            WHERE client_id = ?
            ORDER BY period_end DESC
        ", [$clientId]);
    }

    public static function getBillingRates() {
        $db = Database::getInstance();

        return $db->fetchAll("SELECT * FROM billing_rates");
    }

    public static function updateBillingRate($sampleType, $dailyRate) {
        $db = Database::getInstance();

        return $db->execute(
            "UPDATE billing_rates SET daily_rate = ? WHERE sample_type = ?",
            [$dailyRate, $sampleType]
        );
    }
}
